'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useParams } from 'next/navigation';
import { 
  ArrowLeft,
  ArrowRight,
  User,
  Send,
  Loader2,
  CheckCircle2,
  FileText,
  Shield,
  AlertTriangle,
  FileCheck,
  Truck,
  Package,
  Globe,
  Car,
  Route,
  Train,
  Ship,
  Building2,
  Wrench,
  Sparkles,
  ArrowLeftRight,
  Download,
  ExternalLink,
  Image as ImageIcon,
  File
} from 'lucide-react';
import { useLanguage } from '@/i18n/LanguageContext';
import { LanguageSwitcherSimple } from '@/components/language-switcher';

// 图标映射
const iconMap: Record<string, typeof Truck> = {
  Truck, Car, Route, Train, Ship, Building2, CheckCircle2, 
  Package, Wrench, Sparkles, ArrowLeftRight, FileText, Globe
};

const policyIconMap: Record<string, typeof Shield> = {
  FileCheck,
  Shield,
  AlertTriangle
};

// 双语服务数据结构
interface ServiceDataItem {
  title: { zh: string; ru: string };
  icon: typeof Truck;
  category: { zh: string; ru: string };
  description: { zh: string; ru: string };
  features: { zh: string[]; ru: string[] };
  process: { step: number; title: { zh: string; ru: string }; desc: { zh: string; ru: string } }[];
  policies: { title: { zh: string; ru: string }; content: { zh: string; ru: string }; icon: string }[];
  requirements: { zh: string[]; ru: string[] };
}

// 默认静态数据（当数据库无数据时使用）- 支持双语
const defaultServicesData: Record<string, ServiceDataItem> = {
  'road-transport': {
    title: { zh: '中俄公路运输', ru: 'Автоперевозки Китай-Россия' },
    icon: Truck,
    category: { zh: '公路运输', ru: 'Автоперевозки' },
    description: { 
      zh: '提供中国至俄罗斯的公路跨境运输服务，覆盖主要边境口岸，时效快、成本低、灵活性高。支持整车运输和零担配载，满足不同规模货物需求。', 
      ru: 'Автомобильные перевозки из Китая в Россию через основные погранпереходы. Быстро, выгодно, гибко. Полная и сборная загрузка.' 
    },
    features: { 
      zh: ['时效稳定，全程GPS追踪', '支持整车/零担灵活配载', '满洲里、绥芬河等多口岸通关', '专业报关团队协助清关', '全程货物保险保障'],
      ru: ['Стабильные сроки, GPS-отслеживание', 'Полная и сборная загрузка', 'Маньчжурия, Суйфэньхэ и др.', 'Профессиональное оформление', 'Страхование грузов']
    },
    process: [
      { step: 1, title: { zh: '询价确认', ru: 'Запрос' }, desc: { zh: '确认货物信息、运输路线和报价', ru: 'Подтверждение деталей' } },
      { step: 2, title: { zh: '上门提货', ru: 'Забор груза' }, desc: { zh: '安排车辆上门装货或送货至仓库', ru: 'Погрузка на складе или забор' } },
      { step: 3, title: { zh: '口岸报关', ru: 'Таможня' }, desc: { zh: '办理出口报关手续', ru: 'Экспортное оформление' } },
      { step: 4, title: { zh: '跨境运输', ru: 'Перевозка' }, desc: { zh: '经边境口岸运往俄罗斯', ru: 'Трансграничная перевозка' } },
      { step: 5, title: { zh: '目的地清关', ru: 'Очистка' }, desc: { zh: '办理俄罗斯进口清关', ru: 'Импортное оформление' } },
      { step: 6, title: { zh: '配送签收', ru: 'Доставка' }, desc: { zh: '送货上门并完成签收', ru: 'Доставка и подписание' } }
    ],
    policies: [
      { title: { zh: '出口监管要求', ru: 'Экспорт' }, content: { zh: '需提供商业发票、装箱单、合同等单证；部分商品需出口许可证；危险品需提供MSDS和危险品申报单。', ru: 'Инвойс, упаковочный лист, контракт. Опасные грузы — MSDS.' }, icon: 'FileCheck' },
      { title: { zh: '进口清关政策', ru: 'Импорт' }, content: { zh: '俄罗斯进口需提供EAC认证（欧亚经济联盟认证）；食品、化妆品等需提供卫生证书；动植物产品需检疫证明。', ru: 'Сертификат EAC, санитарный сертификат для продуктов.' }, icon: 'Shield' },
      { title: { zh: '违禁品清单', ru: 'Запрещено' }, content: { zh: '禁止运输武器弹药、毒品、濒危动植物及其制品；锂电池需按危险品申报；液体需提供成分检测报告。', ru: 'Оружие, наркотики, исчезающие виды. Литий — как опасный груз.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['货物信息：名称、数量、重量、体积', '商业发票（中英文/中俄文）', '装箱单', '贸易合同', '特殊商品需提供相应许可证/证书'],
      ru: ['Информация о грузе: название, количество, вес, объем', 'Инвойс (кит./рус.)', 'Упаковочный лист', 'Контракт', 'Лицензии для спецтоваров']
    }
  },
  'car-export': {
    title: { zh: '整车自驾出口', ru: 'Экспорт автомобилей' },
    icon: Car,
    category: { zh: '公路运输', ru: 'Автоперевозки' },
    description: { 
      zh: '为中国汽车制造商和经销商提供整车出口俄罗斯的专业服务，包括运输、报关、认证等一站式解决方案。', 
      ru: 'Профессиональный экспорт автомобилей из Китая в Россию: перевозка, таможня, сертификация.' 
    },
    features: { 
      zh: ['专业整车运输车辆', '支持乘用车、商用车、工程机械', '协助办理EAC认证', '提供OTTS认证咨询', '目的港清关派送'],
      ru: ['Спецавтовозы', 'Легковые, грузовые, спецтехника', 'Сертификация EAC', 'Консультация OTTS', 'Таможня и доставка']
    },
    process: [
      { step: 1, title: { zh: '车辆确认', ru: 'Подтверждение' }, desc: { zh: '确认车型、数量、车辆信息', ru: 'Модель, количество, данные' } },
      { step: 2, title: { zh: '单证准备', ru: 'Документы' }, desc: { zh: '准备车辆出口所需单证', ru: 'Подготовка документов' } },
      { step: 3, title: { zh: '报关出口', ru: 'Экспорт' }, desc: { zh: '办理车辆出口报关手续', ru: 'Таможенное оформление' } },
      { step: 4, title: { zh: '跨境运输', ru: 'Перевозка' }, desc: { zh: '专业运输车辆运送至俄罗斯', ru: 'Перевозка в Россию' } },
      { step: 5, title: { zh: '进口清关', ru: 'Импорт' }, desc: { zh: '办理俄罗斯车辆进口手续', ru: 'Растаможка в РФ' } },
      { step: 6, title: { zh: '交付验收', ru: 'Передача' }, desc: { zh: '车辆交付并进行验收', ru: 'Передача автомобиля' } }
    ],
    policies: [
      { title: { zh: '车辆出口政策', ru: 'Экспорт' }, content: { zh: '需提供车辆合格证、购车发票、车辆一致性证书；新能源车辆需提供电池检测报告。', ru: 'Сертификат соответствия, счет-фактура. Электроавто — отчет о батарее.' }, icon: 'FileCheck' },
      { title: { zh: '俄罗斯进口政策', ru: 'Импорт РФ' }, content: { zh: '车辆需通过EAC认证；需办理OTTS认证（车辆型式批准）；排放需符合欧5标准。', ru: 'Сертификат EAC, одобрение типа ОТТС, Евро-5.' }, icon: 'Shield' },
      { title: { zh: '税费说明', ru: 'Налоги' }, content: { zh: '俄罗斯车辆进口关税根据排量计算；需缴纳增值税(20%)；新能源车享受关税优惠。', ru: 'Таможенная пошлина по объему двигателя, НДС 20%. Электроавто — льготы.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['车辆合格证', '购车发票', '车辆一致性证书', '车辆照片（外观、铭牌、VIN码）', 'EAC认证文件'],
      ru: ['Сертификат качества', 'Счет-фактура', 'Сертификат соответствия', 'Фото авто (VIN)', 'Сертификат EAC']
    }
  },
  'tir-transport': {
    title: { zh: 'TIR运输', ru: 'TIR перевозки' },
    icon: Route,
    category: { zh: '公路运输', ru: 'Автоперевозки' },
    description: { 
      zh: 'TIR（国际公路运输公约）运输服务，简化通关手续，减少边境停留时间，实现快速跨境运输。', 
      ru: 'Перевозки по системе TIR — упрощенная таможня, минимум задержек на границе.' 
    },
    features: { 
      zh: ['TIR单证专业操作', '减少海关查验次数', '缩短边境通关时间', '降低物流成本', '全程可视化追踪'],
      ru: ['Оформление TIR', 'Меньше проверок', 'Быстрая граница', 'Экономия затрат', 'GPS-трекинг']
    },
    process: [
      { step: 1, title: { zh: 'TIR单证办理', ru: 'Оформление TIR' }, desc: { zh: '申请TIR单证及相关手续', ru: 'Получение книжки TIR' } },
      { step: 2, title: { zh: '货物装载', ru: 'Погрузка' }, desc: { zh: '货物装车并施加关封', ru: 'Загрузка и пломбирование' } },
      { step: 3, title: { zh: '起运地报关', ru: 'Экспорт' }, desc: { zh: '办理出口报关手续', ru: 'Экспортное оформление' } },
      { step: 4, title: { zh: 'TIR运输', ru: 'Перевозка' }, desc: { zh: '凭TIR单证快速通过边境', ru: 'Быстрый проезд границы' } },
      { step: 5, title: { zh: '目的地清关', ru: 'Импорт' }, desc: { zh: '到达后办理进口清关', ru: 'Таможня назначения' } },
      { step: 6, title: { zh: '货物交付', ru: 'Доставка' }, desc: { zh: '拆封验货并交付', ru: 'Распломбирование и передача' } }
    ],
    policies: [
      { title: { zh: 'TIR适用范围', ru: 'Сфера TIR' }, content: { zh: '适用于签约国之间的公路运输；中国和俄罗斯均为TIR公约成员国；需使用TIR持证人车辆。', ru: 'Между странами-участницами. Китай и Россия — участники. Требуется TIR-транспорт.' }, icon: 'FileCheck' },
      { title: { zh: '货物限制', ru: 'Ограничения' }, content: { zh: '危险品、冷链货物等特殊货物需特殊申请；部分敏感商品不适用TIR简化通关。', ru: 'Опасные грузы, рефрижераторы — спецразрешение. Чувствительные товары — не упрощенно.' }, icon: 'Shield' },
      { title: { zh: '担保要求', ru: 'Гарантия' }, content: { zh: '需缴纳TIR担保金；担保金额根据运输路线和货物价值确定。', ru: 'Требуется залог TIR. Сумма зависит от маршрута и стоимости.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['TIR单证申请表', '货物商业发票', '装箱单', '运输合同', 'TIR车辆通行证'],
      ru: ['Заявка на TIR', 'Инвойс', 'Упаковочный лист', 'Контракт', 'TIR-сертификат']
    }
  },
  'railway-transport': {
    title: { zh: '铁路境外段', ru: 'Ж/Д в России' },
    icon: Train,
    category: { zh: '铁路运输', ru: 'Ж/Д перевозки' },
    description: { 
      zh: '提供俄罗斯境内铁路运输服务，与俄罗斯铁路公司合作，覆盖俄罗斯主要城市和工业区。', 
      ru: 'Железнодорожные перевозки по России в партнерстве с РЖД. Основные города и промзоны.' 
    },
    features: { 
      zh: ['覆盖俄罗斯全境铁路网', '与俄铁直接合作', '集装箱、整车运输', '稳定班期，准时可靠', '专业铁路团队操作'],
      ru: ['Вся сеть РЖД', 'Прямое партнерство', 'Контейнеры, вагоны', 'Стабильное расписание', 'Профессиональная команда']
    },
    process: [
      { step: 1, title: { zh: '货物委托', ru: 'Заявка' }, desc: { zh: '确认运输计划和报价', ru: 'План и ставка' } },
      { step: 2, title: { zh: '装箱发运', ru: 'Отправка' }, desc: { zh: '货物装箱并运至车站', ru: 'Загрузка и подача' } },
      { step: 3, title: { zh: '铁路运输', ru: 'Перевозка' }, desc: { zh: '经铁路运往目的地', ru: 'Ж/Д до станции' } },
      { step: 4, title: { zh: '到站通知', ru: 'Уведомление' }, desc: { zh: '到达目的地车站通知', ru: 'Прибытие на станцию' } },
      { step: 5, title: { zh: '清关提货', ru: 'Таможня' }, desc: { zh: '办理清关手续提货', ru: 'Очистка и получение' } },
      { step: 6, title: { zh: '配送签收', ru: 'Доставка' }, desc: { zh: '送货至指定地点', ru: 'Доставка до двери' } }
    ],
    policies: [
      { title: { zh: '铁路运输时效', ru: 'Сроки' }, content: { zh: '莫斯科方向约18-22天；圣彼得堡方向约20-25天；其他城市根据具体线路确定。', ru: 'Москва: 18-22 дня. Санкт-Петербург: 20-25 дней. Другие города — по маршруту.' }, icon: 'FileCheck' },
      { title: { zh: '货物限制', ru: 'Ограничения' }, content: { zh: '危险品需按危险品运输规定办理；超大超重货物需提前确认能否运输。', ru: 'Опасные грузы — по правилам. Негабарит — согласование.' }, icon: 'Shield' },
      { title: { zh: '保险建议', ru: 'Страхование' }, content: { zh: '建议为货物购买运输保险；保险费率根据货物种类和价值确定。', ru: 'Рекомендуется страхование. Тариф зависит от груза.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['货物详细清单', '商业发票', '装箱单', '贸易合同', '特殊货物证明文件'],
      ru: ['Подробный список', 'Инвойс', 'Упаковочный лист', 'Контракт', 'Документы для спецгрузов']
    }
  },
  'railway-import': {
    title: { zh: '铁路大宗货物进口', ru: 'Импорт ж/д' },
    icon: Ship,
    category: { zh: '铁路运输', ru: 'Ж/Д перевозки' },
    description: { 
      zh: '专业承接俄罗斯至中国的铁路大宗货物进口运输，包括木材、矿石、煤炭、化肥等原材料。', 
      ru: 'Импорт массовых грузов из России в Китай: лес, руда, уголь, удобрения.' 
    },
    features: { 
      zh: ['大宗货物运输经验', '口岸换装专业操作', '进口清关代理', '国内配送网络', '供应链金融支持'],
      ru: ['Опыт массовых грузов', 'Перегрузка на границе', 'Таможенный брокер', 'Доставка по Китаю', 'Финансирование']
    },
    process: [
      { step: 1, title: { zh: '需求确认', ru: 'Запрос' }, desc: { zh: '确认货物种类、数量和运输需求', ru: 'Тип, объем, требования' } },
      { step: 2, title: { zh: '俄罗斯发运', ru: 'Отправка' }, desc: { zh: '安排俄罗斯境内运输至边境', ru: 'Доставка до границы' } },
      { step: 3, title: { zh: '口岸换装', ru: 'Перегрузка' }, desc: { zh: '在边境口岸进行换装作业', ru: 'Перегрузка на границе' } },
      { step: 4, title: { zh: '进口清关', ru: 'Импорт' }, desc: { zh: '办理进口报关和检验检疫', ru: 'Таможня и карантин' } },
      { step: 5, title: { zh: '国内运输', ru: 'Доставка' }, desc: { zh: '铁路或公路运至目的地', ru: 'Ж/Д или авто до точки' } },
      { step: 6, title: { zh: '货物交付', ru: 'Передача' }, desc: { zh: '货物交付并签收', ru: 'Приемка груза' } }
    ],
    policies: [
      { title: { zh: '进口关税', ru: 'Пошлины' }, content: { zh: '木材、矿石等资源类商品关税较低；化肥等商品需关注配额政策。', ru: 'Лес, руда — низкие пошлины. Удобрения — квоты.' }, icon: 'FileCheck' },
      { title: { zh: '检验检疫', ru: 'Карантин' }, content: { zh: '木材需提供熏蒸证明；矿石需提供成分检测报告；动植物产品需检疫审批。', ru: 'Лес — сертификат фумигации. Руда — анализ состава.' }, icon: 'Shield' },
      { title: { zh: '环保要求', ru: 'Экология' }, content: { zh: '部分矿石需提供放射性检测报告；进口固体废物需特殊许可。', ru: 'Радиационный контроль для руды. Отходы — спецразрешение.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['进口合同', '原产地证明', '商业发票', '装箱单', '品质检验证书'],
      ru: ['Импортный контракт', 'Сертификат происхождения', 'Инвойс', 'Упаковочный лист', 'Сертификат качества']
    }
  },
  'port-agency': {
    title: { zh: '满洲里口岸代理', ru: 'Агент в Маньчжурии' },
    icon: Building2,
    category: { zh: '口岸服务', ru: 'Погранпереход' },
    description: { 
      zh: '满洲里口岸专业代理服务，提供报关报检、口岸换装、仓储配送、单证代办等一站式服务。', 
      ru: 'Агентские услуги на погранпереходе Маньчжурия: таможня, перегрузка, склад, документы.' 
    },
    features: { 
      zh: ['满洲里本地专业团队', '熟悉口岸通关流程', '快速通关通道', '仓储换装设施', '24小时应急响应'],
      ru: ['Местная команда', 'Знание процедур', 'Быстрый проход', 'Склад и перегрузка', '24/7 поддержка']
    },
    process: [
      { step: 1, title: { zh: '委托受理', ru: 'Заявка' }, desc: { zh: '接收运输和报关委托', ru: 'Принятие поручения' } },
      { step: 2, title: { zh: '单证审核', ru: 'Проверка' }, desc: { zh: '审核报关所需单证', ru: 'Проверка документов' } },
      { step: 3, title: { zh: '报关申报', ru: 'Декларация' }, desc: { zh: '向海关申报进出口', ru: 'Подача декларации' } },
      { step: 4, title: { zh: '查验配合', ru: 'Осмотр' }, desc: { zh: '配合海关查验货物', ru: 'Присутствие при досмотре' } },
      { step: 5, title: { zh: '税费缴纳', ru: 'Оплата' }, desc: { zh: '代缴关税和增值税', ru: 'Уплата пошлин' } },
      { step: 6, title: { zh: '放行提货', ru: 'Выпуск' }, desc: { zh: '海关放行后提货发运', ru: 'Выпуск и отправка' } }
    ],
    policies: [
      { title: { zh: '报关时限', ru: 'Сроки' }, content: { zh: '进出口货物申报时限为运输工具申报进出境之日起14日内；逾期可能产生滞报金。', ru: 'Срок декларации — 14 дней. Просрочка — штраф.' }, icon: 'FileCheck' },
      { title: { zh: '查验规定', ru: 'Досмотр' }, content: { zh: '海关可对货物进行查验；查验时货主或代理人应到场；查验不合格将影响通关。', ru: 'Таможня может досмотреть. Владелец должен присутствовать.' }, icon: 'Shield' },
      { title: { zh: '违规处罚', ru: 'Штрафы' }, content: { zh: '申报不实将面临罚款；逃避海关监管可能没收货物；严重违规将追究法律责任。', ru: 'Недостоверная декларация — штраф. Уклонение — конфискация.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['进出口合同', '商业发票', '装箱单', '代理报关委托书', '其他特殊单证（如许可证）'],
      ru: ['Контракт', 'Инвойс', 'Упаковочный лист', 'Доверенность на декларацию', 'Специальные разрешения']
    }
  },
  'customs-clearance': {
    title: { zh: '俄罗斯代理清关', ru: 'Таможня в РФ' },
    icon: CheckCircle2,
    category: { zh: '口岸服务', ru: 'Погранпереход' },
    description: { 
      zh: '俄罗斯境内专业清关代理服务，熟悉俄罗斯海关法规，提供高效合规的清关解决方案。', 
      ru: 'Профессиональная растаможка в России. Знаем таможенные правила.' 
    },
    features: { 
      zh: ['俄罗斯本地清关团队', '熟悉俄海关法规政策', 'EAC认证代办服务', '清关风险把控', '税费合理规划'],
      ru: ['Местная команда', 'Знание законов РФ', 'Сертификация EAC', 'Контроль рисков', 'Оптимизация налогов']
    },
    process: [
      { step: 1, title: { zh: '单证准备', ru: 'Документы' }, desc: { zh: '准备俄罗斯清关所需单证', ru: 'Подготовка документов' } },
      { step: 2, title: { zh: '货物到境', ru: 'Прибытие' }, desc: { zh: '货物到达俄罗斯边境', ru: 'Прибытие на границу' } },
      { step: 3, title: { zh: '申报缴纳', ru: 'Декларация' }, desc: { zh: '申报并缴纳进口税费', ru: 'Подача и оплата' } },
      { step: 4, title: { zh: '海关查验', ru: 'Досмотр' }, desc: { zh: '配合海关查验货物', ru: 'Таможенный досмотр' } },
      { step: 5, title: { zh: '认证核验', ru: 'Сертификаты' }, desc: { zh: '核验EAC等认证文件', ru: 'Проверка EAC' } },
      { step: 6, title: { zh: '放行提货', ru: 'Выпуск' }, desc: { zh: '清关放行后提货配送', ru: 'Выпуск и доставка' } }
    ],
    policies: [
      { title: { zh: '进口关税', ru: 'Пошлины' }, content: { zh: '关税税率根据HS编码确定；部分商品享受最惠国税率；需缴纳20%增值税。', ru: 'Ставка по коду ТН ВЭД. НДС 20%.' }, icon: 'FileCheck' },
      { title: { zh: 'EAC认证', ru: 'EAC' }, content: { zh: '欧亚经济联盟强制性认证；覆盖机械、电子、食品等类别；需在认可实验室检测。', ru: 'Обязательный сертификат ЕАЭС. Машины, электроника, продукты.' }, icon: 'Shield' },
      { title: { zh: '清关风险', ru: 'Риски' }, content: { zh: '单证不齐将无法清关；申报不实将面临罚款；部分商品需进口许可。', ru: 'Нет документов — нет очистки. Недостоверная декларация — штраф.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['商业发票（俄语翻译）', '装箱单（俄语翻译）', '原产地证明', 'EAC认证证书', '进口商注册文件'],
      ru: ['Инвойс (перевод)', 'Упаковочный лист (перевод)', 'Сертификат происхождения', 'Сертификат EAC', 'Регистрация импортера']
    }
  },
  'oversize-transport': {
    title: { zh: '超大件设备运输', ru: 'Негабаритные грузы' },
    icon: Package,
    category: { zh: '特殊物流', ru: 'Спецгрузы' },
    description: { 
      zh: '专业超大件、超重设备运输服务，包括工程机械、生产线设备、能源设备等大型设备的跨境运输。', 
      ru: 'Перевозка негабаритных и тяжеловесных грузов: спецтехника, оборудование, энергетика.' 
    },
    features: { 
      zh: ['特种运输车辆', '大件运输许可办理', '路线勘察规划', '专业装载加固', '全程护送服务'],
      ru: ['Спецтранспорт', 'Разрешения', 'Разработка маршрута', 'Крепление груза', 'Сопровождение']
    },
    process: [
      { step: 1, title: { zh: '设备勘测', ru: 'Обследование' }, desc: { zh: '勘测设备尺寸和重量', ru: 'Замеры и вес' } },
      { step: 2, title: { zh: '方案设计', ru: 'Проект' }, desc: { zh: '设计运输和装载方案', ru: 'План перевозки' } },
      { step: 3, title: { zh: '许可办理', ru: 'Разрешения' }, desc: { zh: '办理大件运输许可证', ru: 'Получение разрешений' } },
      { step: 4, title: { zh: '装载加固', ru: 'Погрузка' }, desc: { zh: '专业装载并加固设备', ru: 'Загрузка и крепление' } },
      { step: 5, title: { zh: '护送运输', ru: 'Перевозка' }, desc: { zh: '专业车辆护送运输', ru: 'Перевозка с сопровождением' } },
      { step: 6, title: { zh: '卸货验收', ru: 'Выгрузка' }, desc: { zh: '目的地卸货并验收', ru: 'Разгрузка и приемка' } }
    ],
    policies: [
      { title: { zh: '超限标准', ru: 'Нормы' }, content: { zh: '超长：长度超过20米；超宽：宽度超过3米；超高：高度超过4.5米；超重：单件超过40吨。', ru: 'Длина > 20м, ширина > 3м, высота > 4.5м, вес > 40т.' }, icon: 'FileCheck' },
      { title: { zh: '许可申请', ru: 'Разрешения' }, content: { zh: '需提前申请超限运输许可证；跨国运输需多国许可；审批周期约15-30天。', ru: 'Требуется спецразрешение. Международные — несколько стран. Срок: 15-30 дней.' }, icon: 'Shield' },
      { title: { zh: '运输限制', ru: 'Ограничения' }, content: { zh: '部分路段限时限行；需配备护送车辆；部分桥梁无法通过超重车辆。', ru: 'Ограничения по времени. Требуется машина сопровождения. Мосты — ограничения.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['设备详细尺寸图', '设备重量分布说明', '设备照片及视频', '起卸货地点信息', '设备价值证明（投保用）'],
      ru: ['Чертежи размеров', 'Распределение веса', 'Фото и видео', 'Адреса загрузки/выгрузки', 'Стоимость для страховки']
    }
  },
  'project-logistics': {
    title: { zh: '工程项目物流', ru: 'Проектная логистика' },
    icon: Wrench,
    category: { zh: '特殊物流', ru: 'Спецгрузы' },
    description: { 
      zh: '为海外工程项目提供全程物流解决方案，包括设备采购运输、现场配送、物资管理等一体化服务。', 
      ru: 'Логистика для зарубежных проектов: закупки, транспортировка, доставка на площадку, управление.' 
    },
    features: { 
      zh: ['项目物流整体规划', '多式联运方案', '现场物流管理', '物资追踪系统', '应急保障机制'],
      ru: ['Планирование логистики', 'Мультимодальные перевозки', 'Логистика на площадке', 'Отслеживание грузов', 'Экстренная поддержка']
    },
    process: [
      { step: 1, title: { zh: '项目评估', ru: 'Оценка' }, desc: { zh: '评估项目物流需求', ru: 'Анализ потребностей' } },
      { step: 2, title: { zh: '方案制定', ru: 'План' }, desc: { zh: '制定整体物流方案', ru: 'Разработка плана' } },
      { step: 3, title: { zh: '采购运输', ru: 'Закупки' }, desc: { zh: '设备物资采购发运', ru: 'Закупка и отправка' } },
      { step: 4, title: { zh: '现场配送', ru: 'Доставка' }, desc: { zh: '物资运至项目现场', ru: 'Доставка на площадку' } },
      { step: 5, title: { zh: '库存管理', ru: 'Склад' }, desc: { zh: '现场物资库存管理', ru: 'Управление запасами' } },
      { step: 6, title: { zh: '项目收尾', ru: 'Завершение' }, desc: { zh: '项目结束物资退运', ru: 'Вывоз материалов' } }
    ],
    policies: [
      { title: { zh: '项目物资出口', ru: 'Экспорт' }, content: { zh: '工程项目物资可申请出口退税；临时出口物资可办理ATA单证册；需提供项目合同证明。', ru: 'Возврат НДС при экспорте. ATA карнет для временного вывоза. Нужен контракт.' }, icon: 'FileCheck' },
      { title: { zh: '临时进口', ru: 'Временный ввоз' }, content: { zh: '工程项目设备可申请临时进口；免缴关税需缴纳保证金；项目结束后需复运出境。', ru: 'Временный ввоз оборудования. Без пошлин с депозитом. Обратный вывоз.' }, icon: 'Shield' },
      { title: { zh: '合规要求', ru: 'Соответствие' }, content: { zh: '需符合两国项目合作规定；部分设备需特殊许可；施工人员需办理工作许可。', ru: 'Соблюдение правил обеих стран. Спецразрешения. Рабочие визы.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['工程项目合同', '设备物资清单', '项目进度计划', '收货人信息', '项目许可文件'],
      ru: ['Контракт проекта', 'Список оборудования', 'График проекта', 'Данные получателя', 'Лицензии проекта']
    }
  },
  'exhibition-logistics': {
    title: { zh: '展会物流', ru: 'Выставочная логистика' },
    icon: Sparkles,
    category: { zh: '特殊物流', ru: 'Спецгрузы' },
    description: { 
      zh: '提供中俄展会物流服务，包括展品运输、现场布展、展后回运等全流程服务。', 
      ru: 'Логистика для выставок Китай-Россия: перевозка экспонатов, монтаж, обратная доставка.' 
    },
    features: { 
      zh: ['ATA单证册代办', '展馆到馆服务', '现场布展支持', '展品仓储服务', '展后回运安排'],
      ru: ['ATA карнет', 'Стенд-в-стенд', 'Помощь с монтажом', 'Склад экспонатов', 'Обратная доставка']
    },
    process: [
      { step: 1, title: { zh: '展品确认', ru: 'Экспонаты' }, desc: { zh: '确认展品清单和信息', ru: 'Список и данные' } },
      { step: 2, title: { zh: 'ATA办理', ru: 'ATA' }, desc: { zh: '办理ATA单证册', ru: 'Оформление ATA' } },
      { step: 3, title: { zh: '展品发运', ru: 'Отправка' }, desc: { zh: '展品运往展馆', ru: 'Доставка на выставку' } },
      { step: 4, title: { zh: '布展支持', ru: 'Монтаж' }, desc: { zh: '协助展品布展', ru: 'Помощь с установкой' } },
      { step: 5, title: { zh: '展期服务', ru: 'Выставка' }, desc: { zh: '展期物流支持', ru: 'Поддержка во время выставки' } },
      { step: 6, title: { zh: '展后回运', ru: 'Возврат' }, desc: { zh: '展品回运或处理', ru: 'Обратная доставка' } }
    ],
    policies: [
      { title: { zh: 'ATA单证册', ru: 'ATA карнет' }, content: { zh: '展会物品可使用ATA单证册临时进出口；免缴关税和保证金；有效期一般为1年。', ru: 'Временный ввоз по ATA карнету. Без пошлин и депозита. Срок действия 1 год.' }, icon: 'FileCheck' },
      { title: { zh: '展品限制', ru: 'Ограничения' }, content: { zh: '消耗品不适用ATA单证册；食品类展品需检验检疫；危险品展品需特殊申请。', ru: 'Расходные материалы — не по ATA. Продукты — карантин. Опасные — спецразрешение.' }, icon: 'Shield' },
      { title: { zh: '时限要求', ru: 'Сроки' }, content: { zh: 'ATA单证册需在有效期内复运出境；逾期未复运将产生关税和罚款。', ru: 'ATA действителен до обратного вывоза. Просрочка — пошлины и штрафы.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['展品清单及价值', '展会邀请函', '参展合同', '展位信息', 'ATA单证册申请表'],
      ru: ['Список и стоимость', 'Приглашение', 'Контракт участника', 'Информация о стенде', 'Заявка на ATA']
    }
  },
  'return-service': {
    title: { zh: '退运服务', ru: 'Возврат грузов' },
    icon: ArrowLeftRight,
    category: { zh: '其他服务', ru: 'Другие услуги' },
    description: { 
      zh: '提供货物退运服务，包括出口货物退运回国、质量问题退货处理等。', 
      ru: 'Услуги по возврату грузов: реэкспорт, возврат по качеству и т.д.' 
    },
    features: { 
      zh: ['退运方案设计', '退运报关代理', '退税款处理', '质量证明代办', '仓储分拣服务'],
      ru: ['План возврата', 'Таможенное оформление', 'Возврат налогов', 'Документы о качестве', 'Склад и сортировка']
    },
    process: [
      { step: 1, title: { zh: '退运确认', ru: 'Подтверждение' }, desc: { zh: '确认退运原因和方案', ru: 'Причина и план' } },
      { step: 2, title: { zh: '单证准备', ru: 'Документы' }, desc: { zh: '准备退运所需单证', ru: 'Подготовка документов' } },
      { step: 3, title: { zh: '退运发运', ru: 'Отправка' }, desc: { zh: '安排货物退运发运', ru: 'Отправка возврата' } },
      { step: 4, title: { zh: '进口报关', ru: 'Импорт' }, desc: { zh: '办理退运进口报关', ru: 'Таможенное оформление' } },
      { step: 5, title: { zh: '税费处理', ru: 'Налоги' }, desc: { zh: '处理退税款或补税', ru: 'Возврат или доплата' } },
      { step: 6, title: { zh: '货物交付', ru: 'Передача' }, desc: { zh: '退运货物交付验收', ru: 'Приемка груза' } }
    ],
    policies: [
      { title: { zh: '退运条件', ru: 'Условия' }, content: { zh: '出口货物因质量、规格等原因退运；需提供原始出口报关单；需提供退运原因证明。', ru: 'Причина: качество, спецификации. Нужна оригинальная декларация и доказательства.' }, icon: 'FileCheck' },
      { title: { zh: '退税处理', ru: 'Налоги' }, content: { zh: '已退税货物退运需补缴退税款；未退税货物可申请退税；需在规定期限内申报。', ru: 'Возвращенный НДС нужно вернуть. Невозвращенный — можно получить. Соблюдайте сроки.' }, icon: 'Shield' },
      { title: { zh: '时限规定', ru: 'Сроки' }, content: { zh: '出口货物退运应在出口之日起1年内办理；逾期需按一般进口处理。', ru: 'Возврат в течение 1 года после экспорта. Просрочка — обычный импорт.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['原始出口报关单', '退运原因说明', '买卖双方协议', '质量检验报告（如需要）', '原退税证明（如已退税）'],
      ru: ['Оригинал экспортной декларации', 'Обоснование возврата', 'Соглашение сторон', 'Акт качества (при необходимости)', 'Подтверждение возврата НДС']
    }
  },
  'ata-carnet': {
    title: { zh: 'ATA单证册', ru: 'ATA карнет' },
    icon: FileText,
    category: { zh: '其他服务', ru: 'Другие услуги' },
    description: { 
      zh: 'ATA单证册申请代办服务，适用于展会、专业设备、商业样品等临时进出口货物。', 
      ru: 'Оформление ATA карнета для выставок, профессионального оборудования, образцов.' 
    },
    features: { 
      zh: ['ATA申请代办', '担保代缴服务', '使用指导服务', '核销销证服务', '延期续签服务'],
      ru: ['Оформление ATA', 'Гарантийный депозит', 'Инструкции по использованию', 'Закрытие карнета', 'Продление']
    },
    process: [
      { step: 1, title: { zh: '申请评估', ru: 'Оценка' }, desc: { zh: '评估是否符合ATA申请条件', ru: 'Проверка условий' } },
      { step: 2, title: { zh: '资料准备', ru: 'Документы' }, desc: { zh: '准备ATA申请所需资料', ru: 'Подготовка документов' } },
      { step: 3, title: { zh: '担保缴纳', ru: 'Депозит' }, desc: { zh: '缴纳ATA担保金', ru: 'Внесение депозита' } },
      { step: 4, title: { zh: '单证领取', ru: 'Получение' }, desc: { zh: '领取ATA单证册', ru: 'Получение карнета' } },
      { step: 5, title: { zh: '通关使用', ru: 'Использование' }, desc: { zh: '使用ATA单证册通关', ru: 'Прохождение таможни' } },
      { step: 6, title: { zh: '核销销证', ru: 'Закрытие' }, desc: { zh: '完成使用后核销销证', ru: 'Закрытие карнета' } }
    ],
    policies: [
      { title: { zh: '适用范围', ru: 'Применение' }, content: { zh: '展会物品、专业设备、商业样品等临时进出口货物；不适用于消耗品和销售品。', ru: 'Выставочные экспонаты, проф. оборудование, образцы. Не для расходников и продаж.' }, icon: 'FileCheck' },
      { title: { zh: '担保要求', ru: 'Гарантия' }, content: { zh: '担保金额一般为货物价值的50%-100%；担保形式包括现金、银行保函等。', ru: 'Депозит 50-100% от стоимости. Формы: наличные, банковская гарантия.' }, icon: 'Shield' },
      { title: { zh: '使用期限', ru: 'Срок действия' }, content: { zh: 'ATA单证册有效期一般为1年；可申请延期；需在有效期内复运出境。', ru: 'Срок действия 1 год. Возможно продление. Обязательный обратный вывоз.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['货物清单及价值', '货物用途说明', '出国/回国行程计划', '申请人身份证明', '担保资金'],
      ru: ['Список и стоимость грузов', 'Назначение грузов', 'План поездки', 'Удостоверение личности', 'Гарантийный депозит']
    }
  },
  'repair-service': {
    title: { zh: '进境修理调试', ru: 'Ремонт и наладка' },
    icon: Wrench,
    category: { zh: '其他服务', ru: 'Другие услуги' },
    description: { 
      zh: '提供设备进境修理调试服务，包括境外设备入境维修、维修后复运出境等。', 
      ru: 'Ввоз оборудования для ремонта и наладки, обратный вывоз после ремонта.' 
    },
    features: { 
      zh: ['保税修理服务', '零部件进口代理', '维修后复运出境', '维修记录管理', '税务优化方案'],
      ru: ['Ремонт под таможенным контролем', 'Импорт запчастей', 'Обратный вывоз', 'Учет ремонтов', 'Налоговая оптимизация']
    },
    process: [
      { step: 1, title: { zh: '维修确认', ru: 'Подтверждение' }, desc: { zh: '确认维修需求和方案', ru: 'План ремонта' } },
      { step: 2, title: { zh: '进境报关', ru: 'Импорт' }, desc: { zh: '办理设备进境报关', ru: 'Таможенное оформление' } },
      { step: 3, title: { zh: '保税维修', ru: 'Ремонт' }, desc: { zh: '在保税状态下进行维修', ru: 'Ремонт под контролем' } },
      { step: 4, title: { zh: '零部件进口', ru: 'Запчасти' }, desc: { zh: '进口维修所需零部件', ru: 'Импорт запчастей' } },
      { step: 5, title: { zh: '维修完成', ru: 'Завершение' }, desc: { zh: '完成维修并测试', ru: 'Тестирование' } },
      { step: 6, title: { zh: '复运出境', ru: 'Вывоз' }, desc: { zh: '设备复运出境', ru: 'Обратный вывоз' } }
    ],
    policies: [
      { title: { zh: '保税维修', ru: 'Бондовый ремонт' }, content: { zh: '设备进境维修可享受保税待遇；无需缴纳进口关税；需缴纳保证金。', ru: 'Ремонт без уплаты пошлин. Требуется депозит.' }, icon: 'FileCheck' },
      { title: { zh: '零部件进口', ru: 'Запчасти' }, content: { zh: '维修用零部件需单独进口申报；消耗的零部件需缴纳关税。', ru: 'Запчасти декларируются отдельно. Расходники — с пошлиной.' }, icon: 'Shield' },
      { title: { zh: '时限要求', ru: 'Сроки' }, content: { zh: '进境维修设备一般需在6个月内复运出境；可申请延期。', ru: 'Обратный вывоз в течение 6 месяцев. Возможно продление.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['设备信息及故障说明', '维修方案', '设备价值证明', '维修合同', '原产地证明'],
      ru: ['Данные оборудования и неисправности', 'План ремонта', 'Стоимость оборудования', 'Контракт на ремонт', 'Сертификат происхождения']
    }
  },
  'leasing-trade': {
    title: { zh: '中俄租赁贸易', ru: 'Лизинг Китай-Россия' },
    icon: Globe,
    category: { zh: '其他服务', ru: 'Другие услуги' },
    description: { 
      zh: '提供中俄租赁贸易物流服务，包括设备租赁运输、租金结算物流支持等。', 
      ru: 'Логистика для лизинговых операций: перевозка оборудования, поддержка расчетов.' 
    },
    features: { 
      zh: ['租赁运输方案', '享受税收优惠', '保险服务支持', '租金结算物流', '合同物流服务'],
      ru: ['Транспорт для лизинга', 'Налоговые льготы', 'Страхование', 'Логистика платежей', 'Контрактная логистика']
    },
    process: [
      { step: 1, title: { zh: '租赁确认', ru: 'Подтверждение' }, desc: { zh: '确认租赁设备和方案', ru: 'Оборудование и условия' } },
      { step: 2, title: { zh: '合同签订', ru: 'Контракт' }, desc: { zh: '签订租赁贸易合同', ru: 'Подписание договора' } },
      { step: 3, title: { zh: '设备发运', ru: 'Отправка' }, desc: { zh: '安排设备运输发运', ru: 'Перевозка оборудования' } },
      { step: 4, title: { zh: '清关交付', ru: 'Доставка' }, desc: { zh: '清关后交付承租方', ru: 'Таможня и передача' } },
      { step: 5, title: { zh: '租赁期服务', ru: 'Сопровождение' }, desc: { zh: '租赁期物流支持', ru: 'Поддержка в период лизинга' } },
      { step: 6, title: { zh: '期满处理', ru: 'Завершение' }, desc: { zh: '租赁期满设备处理', ru: 'Судьба оборудования' } }
    ],
    policies: [
      { title: { zh: '税收优惠', ru: 'Льготы' }, content: { zh: '租赁贸易可享受进口关税分期缴纳；部分设备可申请关税减免。', ru: 'Рассрочка таможенных платежей. Возможны льготы.' }, icon: 'FileCheck' },
      { title: { zh: '保险服务', ru: 'Страхование' }, content: { zh: '建议为租赁设备购买运输保险和财产保险；保险费可计入租赁成本。', ru: 'Рекомендуется страховка. Премия включается в затраты.' }, icon: 'Shield' },
      { title: { zh: '期满处理', ru: 'Окончание' }, content: { zh: '租赁期满可购买、续租或退运；需提前确认期满处理方案。', ru: 'Выкуп, продление или возврат. Заранее согласуйте.' }, icon: 'AlertTriangle' }
    ],
    requirements: { 
      zh: ['租赁合同', '设备清单及价值', '承租方资质证明', '租金支付计划', '期满处理方案'],
      ru: ['Договор лизинга', 'Список и стоимость', 'Документы арендатора', 'График платежей', 'План окончания']
    }
  }
};

interface ServiceContent {
  features?: string[];
  process?: { step: number; title: string; desc: string }[];
  policies?: { title: string; content: string; icon: string }[];
  requirements?: string[];
}

interface DynamicContent {
  id: string;
  title: string;
  description: string | null;
  content: string | ServiceContent | null;
  icon: string | null;
  category: string | null;
  cover_image: string | null;
  attachments: string | Attachment[] | null;
}

interface Attachment {
  name: string;
  url: string;
  type: 'image' | 'document';
}

export default function ServiceDetailPage() {
  const params = useParams();
  const { t, locale } = useLanguage();
  const serviceId = params.id as string;
  
  const [dynamicContent, setDynamicContent] = useState<DynamicContent | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [formData, setFormData] = useState({
    contactName: '',
    contactPhone: '',
    contactEmail: '',
    originCity: '',
    destinationCity: '',
    customsCity: '',
    notes: '',
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [submitStatus, setSubmitStatus] = useState<'idle' | 'success' | 'error'>('idle');
  const [isLoggedIn, setIsLoggedIn] = useState(false);

  // 获取动态内容
  useEffect(() => {
    const fetchContent = async () => {
      try {
        const response = await fetch(`/api/content?id=${serviceId}`);
        const data = await response.json();
        if (response.ok && data.content) {
          setDynamicContent(data.content);
        }
      } catch (error) {
        console.error('Error fetching content:', error);
      } finally {
        setIsLoading(false);
      }
    };
    fetchContent();
  }, [serviceId]);

  // 检查登录状态
  useEffect(() => {
    const checkAuth = async () => {
      try {
        const response = await fetch('/api/auth/me');
        const data = await response.json();
        if (data.authenticated && data.customer) {
          setIsLoggedIn(true);
          setFormData(prev => ({
            ...prev,
            contactName: data.customer.name,
            contactPhone: data.customer.phone,
            contactEmail: data.customer.email || '',
          }));
        }
      } catch {
        // 未登录
      }
    };
    checkAuth();
  }, []);

  // 获取默认静态数据
  const staticData = defaultServicesData[serviceId];
  
  // 解析动态内容 - content 可能是字符串或对象
  const parseJsonSafely = (data: string | ServiceContent | null | undefined): ServiceContent => {
    if (!data) return {};
    if (typeof data === 'string') {
      try {
        return JSON.parse(data);
      } catch {
        return {};
      }
    }
    return data as ServiceContent;
  };
  
  const parsedContent: ServiceContent = parseJsonSafely(dynamicContent?.content ?? null);
  
  // 解析附件
  const parseAttachmentsSafely = (data: string | Attachment[] | null | undefined): Attachment[] => {
    if (!data) return [];
    if (typeof data === 'string') {
      try {
        return JSON.parse(data);
      } catch {
        return [];
      }
    }
    return data as Attachment[];
  };
  
  const attachments: Attachment[] = parseAttachmentsSafely(dynamicContent?.attachments);

  // 辅助函数：获取双语文本
  const getLocalizedText = (text: string | { zh: string; ru: string } | undefined): string => {
    if (!text) return '';
    if (typeof text === 'string') return text;
    return text[locale as 'zh' | 'ru'] || text.zh;
  };

  // 辅助函数：获取双语数组
  const getLocalizedArray = (arr: string[] | { zh: string[]; ru: string[] } | undefined): string[] => {
    if (!arr) return [];
    if (Array.isArray(arr)) {
      // 如果是简单字符串数组，尝试翻译
      return arr.map(item => getLocalizedFeature(item));
    }
    return arr[locale as 'zh' | 'ru'] || arr.zh;
  };

  // 服务特点双语映射
  const featureTranslations: Record<string, { zh: string; ru: string }> = {
    // 数据库中的简化特点
    '简化通关': { zh: '简化通关', ru: 'Упрощенная таможня' },
    '减少查验': { zh: '减少查验', ru: 'Меньше проверок' },
    '快速运输': { zh: '快速运输', ru: 'Быстрая перевозка' },
    'GPS追踪': { zh: 'GPS追踪', ru: 'GPS-отслеживание' },
    '整车/零担': { zh: '整车/零担', ru: 'Целая/сборная машина' },
    '专业报关': { zh: '专业报关', ru: 'Таможенное оформление' },
    'EAC认证': { zh: 'EAC认证', ru: 'Сертификация EAC' },
    'OTTS认证': { zh: 'OTTS认证', ru: 'Сертификация OTTS' },
    '专业运输': { zh: '专业运输', ru: 'Профперевозка' },
    '覆盖全境': { zh: '覆盖全境', ru: 'Вся сеть РЖД' },
    '稳定班期': { zh: '稳定班期', ru: 'Стабильное расписание' },
    '专业团队': { zh: '专业团队', ru: 'Профессиональная команда' },
    '大宗专业': { zh: '大宗专业', ru: 'Опыт массовых грузов' },
    '口岸换装': { zh: '口岸换装', ru: 'Перегрузка на границе' },
    '进口清关': { zh: '进口清关', ru: 'Импортное оформление' },
    '本地团队': { zh: '本地团队', ru: 'Местная команда' },
    '快速通关': { zh: '快速通关', ru: 'Быстрый проход' },
    '24小时响应': { zh: '24小时响应', ru: '24/7 поддержка' },
    '税费规划': { zh: '税费规划', ru: 'Оптимизация налогов' },
    '风险把控': { zh: '风险把控', ru: 'Контроль рисков' },
    '特种车辆': { zh: '特种车辆', ru: 'Спецтранспорт' },
    '许可代办': { zh: '许可代办', ru: 'Разрешения' },
    '专业加固': { zh: '专业加固', ru: 'Крепление груза' },
    '整体规划': { zh: '整体规划', ru: 'Планирование логистики' },
    '多式联运': { zh: '多式联运', ru: 'Мультимодальные перевозки' },
    '现场管理': { zh: '现场管理', ru: 'Логистика на площадке' },
    '退运报关': { zh: '退运报关', ru: 'Таможенное оформление' },
    '退税处理': { zh: '退税处理', ru: 'Возврат налогов' },
    '质量证明': { zh: '质量证明', ru: 'Документы о качестве' },
    '申请代办': { zh: '申请代办', ru: 'Оформление' },
    '担保代缴': { zh: '担保代缴', ru: 'Гарантийный депозит' },
    '核销销证': { zh: '核销销证', ru: 'Закрытие документа' },
    '保税修理': { zh: '保税修理', ru: 'Ремонт под контролем' },
    '零部件进口': { zh: '零部件进口', ru: 'Импорт запчастей' },
    '复运出境': { zh: '复运出境', ru: 'Обратный вывоз' },
    '租赁运输': { zh: '租赁运输', ru: 'Лизинговая логистика' },
    '税收优惠': { zh: '税收优惠', ru: 'Налоговые льготы' },
    '保险服务': { zh: '保险服务', ru: 'Страхование' },
  };

  // 获取服务特点（支持双语）
  const getLocalizedFeature = (feature: string): string => {
    const translation = featureTranslations[feature];
    if (translation) {
      return translation[locale as 'zh' | 'ru'] || translation.zh;
    }
    return feature;
  };

  // 辅助函数：获取双语流程数据
  const getLocalizedProcess = (process: { step: number; title: string | { zh: string; ru: string }; desc: string | { zh: string; ru: string } }[] | undefined) => {
    if (!process) return [];
    return process.map(p => ({
      step: p.step,
      title: getLocalizedText(p.title as any),
      desc: getLocalizedText(p.desc as any)
    }));
  };

  // 辅助函数：获取双语政策数据
  const getLocalizedPolicies = (policies: { title: string | { zh: string; ru: string }; content: string | { zh: string; ru: string }; icon: string }[] | undefined) => {
    if (!policies) return [];
    return policies.map(p => ({
      title: getLocalizedText(p.title as any),
      content: getLocalizedText(p.content as any),
      icon: p.icon
    }));
  };

  // 辅助函数：获取分类显示名称（用于卡片上的分类标签）
  const getCategoryDisplayName = (dbCategory: string | null): string => {
    if (!dbCategory) return '';
    // 如果已经是双语对象格式，直接返回对应语言
    if (typeof dbCategory === 'object') {
      return (dbCategory as any)[locale as 'zh' | 'ru'] || (dbCategory as any).zh;
    }
    // 数据库中的分类值（中文）映射到双语显示
    const categoryMap: Record<string, { zh: string; ru: string }> = {
      '公路运输': { zh: '公路运输', ru: 'Автоперевозки' },
      '铁路运输': { zh: '铁路运输', ru: 'Ж/Д перевозки' },
      '口岸服务': { zh: '口岸服务', ru: 'Погранпереход' },
      '特殊物流': { zh: '特殊物流', ru: 'Спецгрузы' },
      '其他服务': { zh: '其他服务', ru: 'Другие услуги' },
    };
    const mapping = categoryMap[dbCategory];
    return mapping ? mapping[locale as 'zh' | 'ru'] : dbCategory;
  };

  // 合并静态数据和动态数据
  // 优先使用静态翻译数据，如果没有则使用动态数据
  const service = {
    title: getLocalizedText(staticData?.title as any) || dynamicContent?.title || t.services.serviceDetail,
    description: getLocalizedText(staticData?.description as any) || dynamicContent?.description || '',
    category: dynamicContent?.category || getLocalizedText(staticData?.category as any) || '',
    icon: dynamicContent?.icon ? (iconMap[dynamicContent.icon] || Truck) : (staticData?.icon || Truck),
    coverImage: dynamicContent?.cover_image || null,
    features: parsedContent.features?.length ? parsedContent.features : getLocalizedArray(staticData?.features as any),
    process: parsedContent.process?.length ? parsedContent.process : getLocalizedProcess(staticData?.process as any),
    policies: parsedContent.policies?.length ? parsedContent.policies : getLocalizedPolicies(staticData?.policies as any),
    requirements: parsedContent.requirements?.length ? parsedContent.requirements : getLocalizedArray(staticData?.requirements as any),
    attachments,
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setIsSubmitting(true);
    setSubmitStatus('idle');

    try {
      const response = await fetch('/api/orders', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          ...formData,
          notes: `[${service.title}] ${formData.notes}`,
        }),
      });

      const result = await response.json();

      if (response.ok && result.success) {
        setSubmitStatus('success');
        setFormData(prev => ({
          ...prev,
          originCity: '',
          destinationCity: '',
          customsCity: '',
          notes: '',
        }));
      } else {
        setSubmitStatus('error');
      }
    } catch {
      setSubmitStatus('error');
    } finally {
      setIsSubmitting(false);
    }
  };

  if (isLoading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <Loader2 className="w-8 h-8 text-blue-600 animate-spin" />
      </div>
    );
  }

  if (!staticData && !dynamicContent) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <h1 className="text-2xl font-bold text-gray-900 mb-4">{t.services.serviceNotFound}</h1>
          <Link href="/services" className="text-blue-600 hover:underline">{t.services.backToList}</Link>
        </div>
      </div>
    );
  }

  const Icon = service.icon;

  return (
    <div className="min-h-screen bg-gray-50">
      {/* 导航栏 */}
      <nav className="fixed top-0 left-0 right-0 z-50 bg-white/80 backdrop-blur-lg border-b border-gray-100">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center gap-3">
              <Link href="/" className="flex items-center gap-3">
                <div className="relative">
                  <div className="w-11 h-11 bg-white rounded-xl flex items-center justify-center shadow-lg border border-gray-100">
                    <img src="/images/logo.png" alt={t.common.logo} className="w-9 h-9 object-contain" />
                  </div>
                  <div className="absolute -top-1 -right-1 w-3 h-3 bg-green-500 rounded-full border-2 border-white"></div>
                </div>
                <div>
                  <h1 className="text-lg font-bold bg-gradient-to-r from-blue-600 to-indigo-700 bg-clip-text text-transparent">{t.common.logo}</h1>
                  <p className="text-xs text-gray-400 font-medium tracking-wide">{t.common.logoSub}</p>
                </div>
              </Link>
            </div>
            <div className="hidden md:flex items-center gap-1">
              <a href="/#about" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.about}</a>
              <Link href="/services" className="px-4 py-2 text-sm text-blue-600 bg-blue-50 rounded-lg transition-all">{t.nav.services}</Link>
              <Link href="/cases" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.cases}</Link>
              <Link href="/faqs" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.faqs}</Link>
              <a href="/#order" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.order}</a>
              <a href="/#contact" className="px-4 py-2 text-sm text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all">{t.nav.contact}</a>
            </div>
            <div className="flex items-center gap-2">
              <LanguageSwitcherSimple />
              <Link 
                href="/user/orders" 
                className="flex items-center gap-2 px-4 py-2 text-gray-600 hover:text-blue-600 hover:bg-blue-50 rounded-lg transition-all text-sm"
              >
                <User className="w-4 h-4" />
                <span className="hidden sm:inline">{t.nav.userCenter}</span>
              </Link>
              <a 
                href="/#order" 
                className="flex items-center gap-2 px-5 py-2.5 bg-gradient-to-r from-blue-600 to-indigo-600 text-white text-sm font-medium rounded-xl hover:shadow-lg hover:shadow-blue-500/25 hover:-translate-y-0.5 transition-all"
              >
                {t.home.inquiryNow}
                <ArrowRight className="w-4 h-4" />
              </a>
            </div>
          </div>
        </div>
      </nav>

      {/* 头部 */}
      <div className="bg-gradient-to-br from-blue-600 via-blue-700 to-indigo-800 text-white pt-16">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
          {/* 封面图片 */}
          {service.coverImage && (
            <div className="mb-6 rounded-2xl overflow-hidden">
              <img 
                src={service.coverImage} 
                alt={service.title}
                className="w-full h-48 md:h-64 object-cover"
              />
            </div>
          )}
          
          <div className="flex items-center gap-4 mb-4">
            <div className="w-14 h-14 bg-white/20 rounded-2xl flex items-center justify-center backdrop-blur-sm">
              <Icon className="w-7 h-7 text-white" />
            </div>
            <div>
              <span className="text-blue-200 text-sm">{getCategoryDisplayName(service.category)}</span>
              <h1 className="text-3xl font-bold">{service.title}</h1>
            </div>
          </div>
          <p className="text-blue-100 text-lg max-w-3xl">{service.description}</p>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div className="grid lg:grid-cols-3 gap-8">
          {/* 左侧内容 */}
          <div className="lg:col-span-2 space-y-8">
            {/* 服务特点 */}
            {service.features.length > 0 && (
              <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100">
                <h2 className="text-xl font-bold text-gray-900 mb-4">{t.services.serviceFeatures}</h2>
                <div className="grid sm:grid-cols-2 gap-3">
                  {service.features.map((feature, index) => (
                    <div key={index} className="flex items-center gap-3 p-3 bg-green-50 rounded-xl">
                      <CheckCircle2 className="w-5 h-5 text-green-600 flex-shrink-0" />
                      <span className="text-sm text-gray-700">{feature}</span>
                    </div>
                  ))}
                </div>
              </div>
            )}

            {/* 服务流程 */}
            {service.process.length > 0 && (
              <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100">
                <h2 className="text-xl font-bold text-gray-900 mb-6">{t.services.serviceProcess}</h2>
                <div className="relative">
                  <div className="absolute left-5 top-0 bottom-0 w-0.5 bg-blue-100"></div>
                  <div className="space-y-6">
                    {service.process.map((step, index) => (
                      <div key={index} className="flex gap-4">
                        <div className="relative z-10 w-10 h-10 bg-blue-600 rounded-full flex items-center justify-center text-white font-semibold flex-shrink-0">
                          {step.step}
                        </div>
                        <div className="flex-1 pt-2">
                          <h3 className="font-semibold text-gray-900">{step.title}</h3>
                          <p className="text-sm text-gray-500 mt-1">{step.desc}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                </div>
              </div>
            )}

            {/* 进出口监管政策 */}
            {service.policies.length > 0 && (
              <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100">
                <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
                  <Shield className="w-6 h-6 text-blue-600" />
                  {t.services.importExportPolicy}
                </h2>
                <div className="space-y-4">
                  {service.policies.map((policy, index) => {
                    const PolicyIcon = policyIconMap[policy.icon || 'FileCheck'] || FileCheck;
                    return (
                      <div key={index} className="p-4 bg-amber-50 rounded-xl border border-amber-100">
                        <div className="flex items-start gap-3">
                          <div className="w-10 h-10 bg-amber-100 rounded-lg flex items-center justify-center flex-shrink-0">
                            <PolicyIcon className="w-5 h-5 text-amber-600" />
                          </div>
                          <div>
                            <h3 className="font-semibold text-gray-900 mb-1">{policy.title}</h3>
                            <p className="text-sm text-gray-600">{policy.content}</p>
                          </div>
                        </div>
                      </div>
                    );
                  })}
                </div>
              </div>
            )}

            {/* 所需资料 */}
            {service.requirements.length > 0 && (
              <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100">
                <h2 className="text-xl font-bold text-gray-900 mb-4">{t.services.requiredDocuments}</h2>
                <ul className="space-y-2">
                  {service.requirements.map((req, index) => (
                    <li key={index} className="flex items-start gap-2 text-sm text-gray-600">
                      <FileText className="w-4 h-4 text-blue-500 mt-0.5 flex-shrink-0" />
                      {req}
                    </li>
                  ))}
                </ul>
              </div>
            )}

            {/* 相关附件 */}
            {service.attachments.length > 0 && (
              <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100">
                <h2 className="text-xl font-bold text-gray-900 mb-4">{t.services.relatedDocuments}</h2>
                <div className="grid sm:grid-cols-2 gap-3">
                  {service.attachments.map((att, index) => (
                    <a
                      key={index}
                      href={att.url}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="flex items-center gap-3 p-4 bg-gray-50 rounded-xl hover:bg-gray-100 transition-colors"
                    >
                      {att.type === 'image' ? (
                        <div className="w-12 h-12 rounded-lg overflow-hidden">
                          <img src={att.url} alt={att.name} className="w-full h-full object-cover" />
                        </div>
                      ) : (
                        <div className="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                          <File className="w-6 h-6 text-blue-600" />
                        </div>
                      )}
                      <div className="flex-1 min-w-0">
                        <div className="text-sm font-medium text-gray-900 truncate">{att.name}</div>
                        <div className="text-xs text-blue-600 flex items-center gap-1">
                          <Download className="w-3 h-3" />
                          {t.services.clickToDownload}
                        </div>
                      </div>
                      <ExternalLink className="w-4 h-4 text-gray-400" />
                    </a>
                  ))}
                </div>
              </div>
            )}
          </div>

          {/* 右侧询价表单 */}
          <div className="lg:col-span-1">
            <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100 sticky top-4">
              <h2 className="text-xl font-bold text-gray-900 mb-6 flex items-center gap-2">
                <Send className="w-5 h-5 text-blue-600" />
                {t.services.quickInquiry}
              </h2>

              {submitStatus === 'success' ? (
                <div className="text-center py-8">
                  <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle2 className="w-8 h-8 text-green-600" />
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">{t.services.submitSuccess}</h3>
                  <p className="text-gray-500 text-sm mb-4">{t.services.submitSuccessDesc}</p>
                  <button
                    onClick={() => setSubmitStatus('idle')}
                    className="text-blue-600 text-sm hover:underline"
                  >
                    {t.services.inquiryNow}
                  </button>
                </div>
              ) : (
                <form onSubmit={handleSubmit} className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      {t.services.contactName} <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="text"
                      name="contactName"
                      value={formData.contactName}
                      onChange={handleInputChange}
                      required
                      placeholder={t.services.contactNamePlaceholder}
                      className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">
                      {t.services.contactPhone} <span className="text-red-500">*</span>
                    </label>
                    <input
                      type="tel"
                      name="contactPhone"
                      value={formData.contactPhone}
                      onChange={handleInputChange}
                      required
                      placeholder={t.services.contactPhonePlaceholder}
                      className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">{t.services.contactEmail}</label>
                    <input
                      type="email"
                      name="contactEmail"
                      value={formData.contactEmail}
                      onChange={handleInputChange}
                      placeholder={t.services.contactEmailPlaceholder}
                      className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>

                  <div className="grid grid-cols-2 gap-3">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">{t.services.originCity}</label>
                      <input
                        type="text"
                        name="originCity"
                        value={formData.originCity}
                        onChange={handleInputChange}
                        placeholder={locale === 'zh' ? '如：广州' : 'напр. Гуанчжоу'}
                        className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 mb-1">{t.services.destinationCity}</label>
                      <input
                        type="text"
                        name="destinationCity"
                        value={formData.destinationCity}
                        onChange={handleInputChange}
                        placeholder={locale === 'zh' ? '如：莫斯科' : 'напр. Москва'}
                        className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                      />
                    </div>
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">{t.services.customsCity}</label>
                    <input
                      type="text"
                      name="customsCity"
                      value={formData.customsCity}
                      onChange={handleInputChange}
                      placeholder={locale === 'zh' ? '如：后贝加尔' : 'напр. Забайкальск'}
                      className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500"
                    />
                  </div>

                  <div>
                    <label className="block text-sm font-medium text-gray-700 mb-1">{t.services.notes}</label>
                    <textarea
                      name="notes"
                      value={formData.notes}
                      onChange={handleInputChange}
                      rows={3}
                      placeholder={t.services.notesPlaceholder}
                      className="w-full px-4 py-2 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-blue-500 resize-none"
                    />
                  </div>

                  {submitStatus === 'error' && (
                    <div className="p-3 bg-red-50 text-red-600 text-sm rounded-xl">
                      {t.services.submitError}
                    </div>
                  )}

                  <button
                    type="submit"
                    disabled={isSubmitting}
                    className="w-full py-3 bg-gradient-to-r from-blue-600 to-indigo-600 text-white font-medium rounded-xl hover:from-blue-700 hover:to-indigo-700 disabled:opacity-50 transition-all flex items-center justify-center gap-2"
                  >
                    {isSubmitting ? (
                      <>
                        <Loader2 className="w-5 h-5 animate-spin" />
                        {t.services.submitting}
                      </>
                    ) : (
                      <>
                        <Send className="w-5 h-5" />
                        {t.services.submit}
                      </>
                    )}
                  </button>

                  {!isLoggedIn && (
                    <p className="text-xs text-gray-400 text-center">
                      {locale === 'zh' ? '已有账号？' : 'Есть аккаунт?'}<Link href="/auth" className="text-blue-600 hover:underline">{locale === 'zh' ? '登录' : 'Войти'}</Link>{locale === 'zh' ? '可查看订单' : 'для просмотра заказов'}
                    </p>
                  )}
                </form>
              )}
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
